from django.contrib.auth import logout
from django.db import transaction
from django.http import JsonResponse
from django.shortcuts import render, get_object_or_404, redirect
from django.contrib.auth.decorators import login_required
from django.views.decorators.http import require_http_methods
from .models import Device, DeviceTelemetry, Refill, Tenant
import json
import string
import secrets
from django.db.models import Sum, Count, Q
from django.db.models.functions import TruncDay, TruncWeek, TruncMonth, TruncYear, ExtractWeekDay
from django.utils import timezone
from datetime import timedelta
from collections import defaultdict
from django.views.decorators.csrf import csrf_exempt

def logout_view(request):
    logout(request)
    return redirect('login')

@csrf_exempt
@require_http_methods(["POST"])
def device_telemetry(request):
    device = request.device
    try:
        data = json.loads(request.body)
        with transaction.atomic():
            DeviceTelemetry.objects.create(
                device=device,
                coffee_dispense_count=data.get('coffee_dispense_count'),
                tea_dispense_count=data.get('tea_dispense_count'),
                water_level=data.get('water_level'),
                temperature=data.get('temperature'),
                status=data.get('status')
            )
            device.coffee_dispense_count += data.get('coffee_dispense_count')
            device.tea_dispense_count += data.get('tea_dispense_count')
            device.water_level = data.get('water_level')
            device.temperature = data.get('temperature')
            device.status = data.get('status')
            if device.coffee_dispense_count >= device.coffee_limit_per_refill or \
               device.tea_dispense_count >= device.tea_limit_per_refill:
                device.status = Device.Status.DISABLED
            device.save()
    except json.JSONDecodeError:
        return JsonResponse({'error': 'Invalid JSON'}, status=400)
    except Exception as e:
        return JsonResponse({'error': str(e)}, status=500)
    return JsonResponse({'status': 'ok'})


@require_http_methods(["GET"])
def device_configuration(request):
    device = request.device
    return JsonResponse({
        'coffee_limit': device.coffee_limit_per_refill,
        'tea_limit': device.tea_limit_per_refill,
        'temperature_limit_H': device.temperature_limit_H,
        'temperature_limit_L': device.temperature_limit_L,
        'activation': device.activation,
        'status': device.status
    })


@login_required
def comprehensive_stats_api(request):
    period = request.GET.get('period', 'month')

    device_activity = Device.objects.annotate(total_dispenses=Sum('coffee_dispense_count') + Sum('tea_dispense_count')).order_by('-total_dispenses')
    top_5_active = list(device_activity[:5].values('name', 'total_dispenses', 'status'))
    bottom_5_active = list(device_activity.order_by('total_dispenses')[:5].values('name', 'total_dispenses'))
    dispenses_by_weekday = DeviceTelemetry.objects.annotate(weekday=ExtractWeekDay('timestamp')).values('weekday').annotate(total=Sum('coffee_dispense_count') + Sum('tea_dispense_count')).order_by('weekday')
    low_water_devices = list(Device.objects.filter(water_level__lt=25).values('name', 'water_level').order_by('water_level'))
    error_by_tenant = Device.objects.filter(status='ERROR').values('tenant__name').annotate(error_count=Count('id')).order_by('-error_count')
    total_consumption = Device.objects.aggregate(total_coffee=Sum('coffee_dispense_count'), total_tea=Sum('tea_dispense_count'))
    total_consumption['total_coffee'] = total_consumption['total_coffee'] or 0
    total_consumption['total_tea'] = total_consumption['total_tea'] or 0
    total_devices = Device.objects.count()
    active_devices = Device.objects.filter(activation='ACTIVE').count()
    error_devices = Device.objects.filter(status='ERROR').count()

    trunc_kind = {'day': TruncDay, 'week': TruncWeek, 'month': TruncMonth, 'year': TruncYear}.get(period, TruncMonth)
    time_delta_options = {'day': timedelta(days=30), 'week': timedelta(weeks=26), 'month': timedelta(days=365)}
    time_delta = time_delta_options.get(period)

    base_query = DeviceTelemetry.objects.all()
    if time_delta:
        start_date = timezone.now() - time_delta
        base_query = base_query.filter(timestamp__gte=start_date)

    performance_data = base_query.annotate(
        period_start=trunc_kind('timestamp')
    ).values('period_start').annotate(
        total_dispenses=Sum('coffee_dispense_count') + Sum('tea_dispense_count')
    ).order_by('period_start')

    performance_chart_data = [
        {'period': p['period_start'].strftime('%Y-%m-%d'), 'total_dispenses': p['total_dispenses']}
        for p in performance_data
    ]
    
    tenant_data = Device.objects.values('tenant__name').annotate(total_dispenses=Sum('coffee_dispense_count') + Sum('tea_dispense_count')).order_by('-total_dispenses')
    return JsonResponse({
        'kpis': {'total_devices': total_devices, 'total_dispenses': total_consumption.get('total_coffee', 0) + total_consumption.get('total_tea', 0), 'active_devices': active_devices, 'error_devices': error_devices,},
        'top_5_active': top_5_active, 'bottom_5_active': bottom_5_active, 'dispenses_by_weekday': list(dispenses_by_weekday),
        'low_water_devices': low_water_devices, 'error_by_tenant': list(error_by_tenant), 'total_consumption': total_consumption,
        'performance_data': performance_chart_data, 
        'dispenses_by_tenant': list(tenant_data)
    })


@login_required
def dashboard(request):
    return render(request, 'clkbx/dashboard.html')


@login_required
def devices(request):
    tenants = Tenant.objects.all() if request.user.is_authenticated and request.user.role == 'SUPER_ADMIN' else []
    return render(request, 'clkbx/devices.html', {'tenants': tenants, 'user_role': request.user.role if request.user.is_authenticated else ''})


@login_required
def reports(request):
    return render(request, 'clkbx/reports.html')


@login_required
@require_http_methods(["GET", "POST"])
def device_api(request):
    user = request.user
    if request.method == 'GET':
        base_query = Device.objects.select_related('tenant').all() if user.role == 'SUPER_ADMIN' else Device.objects.select_related('tenant').filter(tenant=user.tenant)
        if (status := request.GET.get('status')): base_query = base_query.filter(status=status)
        if (activation := request.GET.get('activation')): base_query = base_query.filter(activation=activation)
        if (tenant_id := request.GET.get('tenant_id')) and user.role == 'SUPER_ADMIN': base_query = base_query.filter(tenant_id=tenant_id)
        if (search_term := request.GET.get('search')):
            base_query = base_query.filter(Q(name__icontains=search_term) | Q(tenant__name__icontains=search_term))
        sort_by = request.GET.get('sort_by', 'name')
        if request.GET.get('order', 'asc') == 'desc': sort_by = f'-{sort_by}'
        base_query = base_query.order_by(sort_by)
        devices = list(base_query.values('id', 'name', 'status', 'activation', 'tenant__name', 'temperature', 'water_level'))
        return JsonResponse(devices, safe=False)
    elif request.method == 'POST':
        try:
            data = json.loads(request.body)
            tenant = None
            if user.role == 'SUPER_ADMIN':
                if new_tenant_name := data.get('new_tenant_name'):
                    tenant, created = Tenant.objects.get_or_create(name=new_tenant_name)
                elif tenant_id := data.get('tenant_id'):
                    tenant = get_object_or_404(Tenant, id=tenant_id)
            else:
                tenant = user.tenant
            if not tenant: return JsonResponse({'error': 'Tenant could not be determined.'}, status=400)
            
            Device.objects.create(
                tenant=tenant, name=data.get('name'), api_key=''.join(secrets.choice(string.ascii_letters + string.digits) for _ in range(32)),
                status=data.get('status', 'IDLE'), activation=data.get('activation', 'ACTIVE'),
                coffee_limit_per_refill=data.get('coffee_limit_per_refill', 100), tea_limit_per_refill=data.get('tea_limit_per_refill', 100),
                temperature_limit_H=data.get('temperature_limit_H', 90.0), temperature_limit_L=data.get('temperature_limit_L', 60.0)
            )
            return JsonResponse({'status': 'ok'}, status=201)
        except Exception as e:
            return JsonResponse({'error': str(e)}, status=500)


@login_required
@require_http_methods(["GET", "PUT", "DELETE"])
def device_detail_api(request, device_id):
    device = get_object_or_404(Device, id=device_id)
    if request.method == 'GET':
        return JsonResponse({
            'id': device.id, 'name': device.name, 'api_key': device.api_key, 'status': device.status, 'activation': device.activation,
            'coffee_limit_per_refill': device.coffee_limit_per_refill, 'tea_limit_per_refill': device.tea_limit_per_refill,
            'temperature_limit_H': device.temperature_limit_H, 'temperature_limit_L': device.temperature_limit_L,
        })
    elif request.method == 'PUT':
        try:
            data = json.loads(request.body)
            device.name = data.get('name', device.name)
            device.status = data.get('status', device.status)
            device.activation = data.get('activation', device.activation)
            # Other fields...
            device.save()
            return JsonResponse({'status': 'ok'})
        except Exception as e:
            return JsonResponse({'error': str(e)}, status=500)
    elif request.method == 'DELETE':
        device.delete()
        return JsonResponse({}, status=204)


@login_required
@require_http_methods(["POST"])
def generate_report(request):
    try:
        data = json.loads(request.body)
        telemetry = DeviceTelemetry.objects.filter(
            device_id__in=data.get('device_ids'),
            timestamp__range=[data.get('start_date'), data.get('end_date')]
        ).values('device__name').annotate(total_coffee=Sum('coffee_dispense_count'), total_tea=Sum('tea_dispense_count'))
        return JsonResponse(list(telemetry), safe=False)
    except Exception as e:
        return JsonResponse({'error': str(e)}, status=500)