from django.contrib.auth.models import AbstractUser
from django.db import models
from django.utils.translation import gettext_lazy as _
import uuid

class Tenant(models.Model):
    id = models.UUIDField(primary_key=True, default=uuid.uuid4, editable=False)
    name = models.CharField(max_length=255)
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)

    def __str__(self):
        return self.name

class User(AbstractUser):
    class Role(models.TextChoices):
        SUPER_ADMIN = 'SUPER_ADMIN', _('Super Admin')
        ADMIN = 'ADMIN', _('Admin')
        USER = 'USER', _('User')

    id = models.UUIDField(primary_key=True, default=uuid.uuid4, editable=False)
    tenant = models.ForeignKey(Tenant, on_delete=models.CASCADE, null=True, blank=True)
    role = models.CharField(max_length=50, choices=Role.choices, default=Role.USER)

    def __str__(self):
        return self.username

class Device(models.Model):
    class Status(models.TextChoices):
        IDLE = 'IDLE', _('Idle')
        DISPENSING = 'DISPENSING', _('Dispensing')
        ERROR = 'ERROR', _('Error')
        DISABLED = 'DISABLED', _('Disabled')

    class Activation(models.TextChoices):
        ACTIVE = 'ACTIVE', _('Active')
        INACTIVE = 'INACTIVE', _('Inactive')

    id = models.UUIDField(primary_key=True, default=uuid.uuid4, editable=False)
    tenant = models.ForeignKey(Tenant, on_delete=models.CASCADE)
    name = models.CharField(max_length=255)
    api_key = models.CharField(max_length=255, unique=True)
    status = models.CharField(max_length=50, choices=Status.choices, default=Status.IDLE)
    activation = models.CharField(max_length=50, choices=Activation.choices, default=Activation.ACTIVE)
    coffee_limit_per_refill = models.PositiveIntegerField(default=100)
    tea_limit_per_refill = models.PositiveIntegerField(default=100)
    temperature_limit_H = models.FloatField(default=90.0)
    temperature_limit_L = models.FloatField(default=60.0)
    coffee_dispense_count = models.PositiveIntegerField(default=0)
    tea_dispense_count = models.PositiveIntegerField(default=0)
    water_level = models.PositiveIntegerField(default=100)
    temperature = models.FloatField(default=0.0)
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)

    def __str__(self):
        return self.name

class DeviceTelemetry(models.Model):
    id = models.UUIDField(primary_key=True, default=uuid.uuid4, editable=False)
    device = models.ForeignKey(Device, on_delete=models.CASCADE)
    timestamp = models.DateTimeField(auto_now_add=True)
    coffee_dispense_count = models.PositiveIntegerField()
    tea_dispense_count = models.PositiveIntegerField()
    water_level = models.PositiveIntegerField()
    temperature = models.FloatField()
    status = models.CharField(max_length=50, choices=Device.Status.choices)

    def __str__(self):
        return f'{self.device.name} - {self.timestamp}'

class Refill(models.Model):
    id = models.UUIDField(primary_key=True, default=uuid.uuid4, editable=False)
    device = models.ForeignKey(Device, on_delete=models.CASCADE)
    user = models.ForeignKey(User, on_delete=models.CASCADE)
    timestamp = models.DateTimeField(auto_now_add=True)

    def __str__(self):
        return f'{self.device.name} - {self.timestamp}'